namespace org.owasp.appsensor.filters
{
    using System;
    using System.Collections.Generic;
    using System.Web;
    using System.Web.SessionState;
    using org.owasp.appsensor.errors;

    /**
     * This class is a servlet filter that can be configured in the web.xml to 
     * detect if the IP address associated with a given session id changes mid-session.  
     * This could point to an attacker trying to reuse a session id from another machine.
     * 
     * The configuration in web.xml should be setup like this.
     * 
     * <filter>
     *    	<filter-name>IpAddressChangeDetectionFilter</filter-name>
     *    	<filter-class>
     *			org.owasp.appsensor.filters.IpAddressChangeDetectionFilter
     *		</filter-class>
     * 	</filter>
     * 	<filter-mapping>
     *	    <filter-name>IpAddressChangeDetectionFilter</filter-name>
     *	    <url-pattern>/*</url-pattern>
     * 	</filter-mapping>
     * 
     */
public class IpAddressChangeDetectionFilter : IHttpModule {

	/** Map that holds mappings of session id's to client IP addresses. */
	//jtm - 5/23/2011 - change to concurrent map to address http://code.google.com/p/appsensor/issues/detail?id=6
	private static Dictionary<String,String> sessionClientIPMap = new Dictionary<String,String>();
	
	/**
	 * Default constructor.
	 */
	public IpAddressChangeDetectionFilter() {
	}

    public void Dispose()
    {
    }

    public void Init(HttpApplication context)
    {
        HttpRequest request = context.Request;

        HttpSessionState session = context.Session;
        String clientIP = request.ServerVariables["REMOTE_ADDR"];

        //check if we can proceed
        if (session != null && session.SessionID != null && clientIP != null)
        {
            String sessionId = session.SessionID;

            //if session id is new to tracking - just add it to map
            if (!sessionClientIPMap.ContainsKey(sessionId))
            {
                if ( !sessionClientIPMap.ContainsKey(sessionId) ) sessionClientIPMap.Add(sessionId, clientIP);
            }
            else
            {
                //session id is already in map - check if clientIP matches
                String existingClientIP = sessionClientIPMap[sessionId];

                //if there is a discrepancy, report the exception
                if (!clientIP.Equals(existingClientIP))
                {
                    new AppSensorException("SE5", "AppSensorUser Message SE5",
                            "Client IP Address for session has changed - original client IP [" + existingClientIP + "]" +
                            " / new client IP [" + clientIP + "] for session id [" + sessionId + "]");
                }
            }

        }
    }
}
}